/**
 * \file keyman_common.h
 *
 * \brief Definition of the functions which might be used by any architecture
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __LIBKEYMAN_INTERN_H_
#define __LIBKEYMAN_INTERN_H_

#include <private/keyman_arch.h>

/**
 * \ingroup libKeyman
 * \defgroup libKeyman_Intern_API libKeyman internal API
 *
 * Common functions which can be used by the architecture specific code
 */

/**
 * \ingroup libKeyman_Intern_API
 * \brief Used to specify that this function is part of the internal API of libKeyman
 *
 * These function will be visible to applications including libkeyman library */
#define KEYMAN_INTERN_API __attribute__((__visibility__("internal")))

/**
 * \ingroup libKeyman_Intern_API
 * \brief Determine the initial value used when iterating over
 * the KID range defined by \p first and \p last
 *
 * Using \ref keyman_intern_search_kid_init and
 * \ref keyman_intern_search_kid_next it is possible to iterate over
 * the complete range defined by \p first and \p last.
 * A random value within the range will be used as the starting point.
 * In case \p first == \p last this is the initial/only valid KID
 *
 *
 * \param[in] first lower border of the KID range
 * \param[in] last upper border of the KID range
 * \param[out] init_kid pointer to return the initial KID
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_UNKNOWN_ERROR - failed to generate random
 * \return else appropriate error value
 */
KEYMAN_INTERN_API sdc_error_t keyman_intern_search_kid_init(sdc_key_id_t first,
                                                            sdc_key_id_t last, sdc_key_id_t *init_kid);

/**
 * \ingroup libKeyman_Intern_API
 * Determine the next KID when iterating over
 * the KID range defined by \p first and \p last
 *
 * See \ref keyman_intern_search_kid_init
 *
 * \param[in] first lower border of the KID range
 * \param[in] last upper border of the KID range
 * \param[in] init_kid KID returned by \ref keyman_intern_search_kid_init
 * \param[out] kid pointer to return the next KID
 *
 * \return \ref SDC_OK on success
 * \return \ref SDC_KID_EXISTS - no other value in the defined range
 * \return else appropriate error value
 */
KEYMAN_INTERN_API sdc_error_t keyman_intern_search_kid_next(sdc_key_id_t first,
                                                            sdc_key_id_t last, sdc_key_id_t init_kid, sdc_key_id_t *kid);

#endif
